/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

#ifndef IMSLIB_H
#define IMSLIB_H

#ifdef MAC
	#define EXPORT extern "C"
#else
	#define EXPORT extern "C" __declspec(dllexport)
#endif
extern "C"
{
	/**
	 * IMSLib Status enum
	 * Zero status code implies Sucess.
	 * Non zero status code implies some error.
	 */
	typedef enum IMS_STATUS
	{
		IMS_SUCCESS = 0, /**< General Success status.*/
		IMS_ERROR_FAILURE = 1,/**< General failure status.*/
		IMS_ERROR_INVALID_ARGUMENTS = 2, /**< Error returned if invalid arguments are passed to an API.*/
		
		IMS_ERROR_CANCEL = 20, /**< Error returned if the Http request is cancelled.*/
		IMS_ERROR_TIMEOUT = 21,/**< Error returned if the Http request is timed out.*/
		IMS_ERROR_HTTPFAILURE = 22, /**< Error returned if http request fails.*/
		IMS_ERROR_SSLFAILURE = 23,/**< Error returned if http request fails, due to SSL certifiaction error.*/
		IMS_ERROR_AUTH_PROXY_REQUIRED = 24, /**< Error returned if http request is being made behind proxy.*/
		IMS_ERROR_AUTH_PROXY_FAILED = 25, /**< Error returned if http request failed due to wrong proxy credentials.*/
		IMS_ERROR_IN_ACCESS_IDP = 26, /**< Error returned if IMSLib fails in accessing IDP server. Probably server is down.*/
		
		IMS_ERROR_ANOTHER_REQUEST_IN_PROCESS = 40, /**< Error returned if async call to IMSLib is made when one async http request is already in progress.*/

		IMS_ERROR_IN_READ_USER_DATA = 60, /**< Error returned if IMSLib fails in reading user specific data from the user machine.*/
		IMS_ERROR_IN_SAVE_USER_DATA = 61, /**< Error returned if IMSLib fails in saving user specific data on the user machine.*/
		IMS_ERROR_IN_REMOVE_USER_DATA = 62, /**< Error returned if IMSLib fails in removing user specific data from the user machine.*/
		IMS_ERROR_USER_DATA_NOT_PRESENT = 63, /**< Error returned if IMSLib detects that user data is not present on the user machine.*/
		IMS_ERROR_IN_READ_DEVICE_TOKEN= 64, /**< Error returned if IMSLib fails in reading device token from the user machine.*/
		IMS_ERROR_IN_SAVE_DEVICE_TOKEN = 65, /**< Error returned if IMSLib fails in saving device token on the user machine.*/
		IMS_ERROR_IN_REMOVE_DEVICE_TOKEN = 66, /**< Error returned if IMSLib fails in removing device token from the user machine.*/
		IMS_ERROR_DEVICE_TOKEN_NOT_PRESENT = 67, /**< Error returned if device token is not present on the user machine.*/
		IMS_ERROR_INVALID_DEVICE_TOKEN = 68, /**< Error returned if invalid device token is present on the user machine.*/
		IMS_ERROR_CLIENTID_NOT_PRESENT = 69, /**< Error returned, if client id information is missing on the user machine.*/
		IMS_ERROR_IN_FETCH_USER_ACCOUNTS = 70, /**< Error returned, if IMSLib fails in fetching user accounts details from the user machine.*/
		IMS_ERROR_IN_SAVE_USER_FOR_CLIENTID = 71, /**< Error returned, if IMSLib fails in saving user id for a particular client id.*/
		IMS_ERROR_DEVICE_ID_NOT_PRESENT = 72, /**< Error returned if device id is not present on the user machine.*/
		IMS_ERROR_DEFAULT_USER_FOR_CLIENTID_NOT_PRESENT = 73, /**< Error returned if there is no default user for client_id*/
		IMS_ERROR_NO_USER_RECORDS_PRESENT = 74, /**< Error returned if there is no user records are present in opm.db*/
		
		IMS_ERROR_ACCESS_DENIED = 80, /**< Error returned if IMS server returns error code 'access_denied' while generating token.*/
		IMS_ERROR_INVALID_REQUEST = 81, /**< Error returned if IMS server returns error code 'invalid_request' while generating token.*/
		IMS_ERROR_INVALID_CLIENT = 82, /**< Error returned if IMS server returns error code 'invalid_client' while generating token.*/
		IMS_ERROR_INVALID_GRANT = 83, /**< Error returned if IMS server returns error code 'invalid_grant' while generating token.*/
		IMS_ERROR_UNAUTHORIZED_CLIENT = 84, /**< Error returned if IMS server returns error code 'unauthorized_client' while generating token.*/
		IMS_ERROR_UNSUPPORTED_RESPONSE_TYPE = 85, /**< Error returned if IMS server returns error code 'unsupported_response_type' while generating token.*/
		IMS_ERROR_INVALID_SCOPE = 86, /**< Error returned if IMS server returns error code 'invalid_scope' while generating token.*/
		IMS_ERROR_UNSUPPORTED_GRANT_TYPE = 87, /**< Error returned if IMS server returns error code 'unsupported_grant_type' while generating token.*/
		IMS_ERROR_BAD_REQUEST = 88, /**< Error returned if IMS server returns error code 'bad_request' while generating token.*/
		IMS_ERROR_INVALID_CREDENTIALS = 89, /**< Error returned if IMS server returns error code 'invalid_credentials' while generating token.*/

		IMS_ERROR_IN_GET_AUTH_DATA_FROM_IDP = 100, /**< Error returned if IMSLib fails fetching authentication data from server.*/
		IMS_ERROR_IN_GET_DEVICE_TOKEN_FROM_IDP = 101, /**< Error returned if IMSLib fails fetching device token from server.*/
		IMS_ERROR_IN_GET_REFRESH_TOKEN_FROM_IDP = 102, /**< Error returned if IMSLib fails fetching device token from server.*/
		IMS_ERROR_IN_GET_ACCESS_TOKEN_FROM_IDP = 103, /**< Error returned if IMSLib fails fetching device token from server.*/
		IMS_ERROR_IN_GET_PROFILE_DATA_FROM_IDP = 104, /**< Error returned if IMSLib fails fetching device token from server.*/
		
		IMS_ERROR_TOU_CHANGED = 120, /**< Error returned if TOU has changed. */
		IMS_ERROR_IN_REVOKE_DEVICE_TOKEN = 121, /**< Error returned if IMSLib fails in revoking device token. */
		IMS_ERROR_TOU_NOT_CURRENT = 122, /**< Terms of use acceptance is not current. */
		IMS_ERROR_EVS_INVALID = 123, /**< Email Verification Status does not meet required policy. */
		IMS_ERROR_ACCT_ACT_REQ = 124, /**< Some issue with the user account requires attention/action. i.e. COPPA or some other issue requiring user action, related to the user account rather than a subscription. */
		IMS_ERROR_ACCT_DISABLED = 125, /**< Account is currently disabled. */
		IMS_ERROR_SUBS_ACT_REQ = 126, /**< Some issue with the subscription requires attention/action. i.e. payment error etc. */
		IMS_ERROR_SUBS_NO_SUB= 127, /**< No current subscription exists for the user. i.e. closed/removed. */

		IMS_ERROR_NO_BUDDY_GROUP_FOR_CLIENT = 150, /**< No buddy_group exists for the client queried. */
		IMS_ERROR_CLIENT_REGISTERED_FOR_OTHER_GROUP = 151, /**< Client is already registered for some group other than requested. */
		IMS_ERROR_GROUP_ENTRY_NOT_PRESENT = 152, /**< No entry exixts for the group queried. */
		IMS_ERROR_IN_SAVE_GROUP_DATA = 153, /**< Error Occured while saving the buddy-group data */

		IMS_ERROR_CNAME_ENTRY_NOT_PRESENT = 154,/**< CName entry not present */
		IMS_ERROR_IN_SAVE_BACKOFF_DATA = 155,/**< Error Occured while saving the backoff data for the cname */
		
		/**< For HTTP errors 401 to 600, IMSLib returns them as it is, they are not wrapped by any IMSLib errors*/

		IMS_UNKNOWN_ERROR = 0xFFFF, /**< Unknown error status. */
	} IMS_Status;/**< IMS_Status type. Used as a return value for IMSLib interface methods.*/

	typedef enum IMS_Data_Type
	{
		IMS_TYPE_ACCOUNTS = 0,
		IMS_TYPE_PROXY_DATA = 1,
		IMS_TYPE_DEVICE_ID = 2,
		IMS_TYPE_DEFAULT_USER_DATA = 3,
		IMS_TYPE_UNKNOWN_DATA = 0xFFFF,
	}IMS_Data_Type;

	typedef enum IMS_Buddy_Group
	{
		IMS_BUDDY_GROUP_AAM = 0,
		
		IMS_MAX_GROUP_ID,
	}IMS_Buddy_Group;

	/**
	 * Description: IMSRef is the refrence pointer for IMSLib
	 */
	typedef void* IMSRef;

	/**
	 * Description: This is the function pointer type which will be input to the asynchronous IMSLib APIs
	 *
	 * @param inStatus is the IMS_Status code. It is IMS_SUCCESS in case of success else corresponding error code.
	 * @param inDetails is the JSON format char* data
	 * @param inClientContext is the context of client which is pased in the createRef API.
	 *
	 * @return void
	 *
	 */
	typedef void (*HTTPCallbackFn)( const IMS_Status inStatus, const char* inDetails, void* const inClientContext);

}

	//The usage of IMS_createRef() is not recommended and might soon be deprecated, so use IMS_createRefEx()/IMS_createRefForProductVersion() instead	
	/**
	 *
	 * Description:  This API will return the reference for IMSLib. It will initialize IMSLib, this API will be first call for all the modules that will be using IMSLib.
	 *
	 * @param  inClientContext is the context of the client, which will be maintained in IMSRef and later returned in the HTTPCallbackFn (REQUIRED).
	 * @param  inIsGM is the bool parameter to decide whether IMSLib needs interface with IMS's production or pre-release end points. 
	 *			If inIsGM is true IMSLib will use production endpoint of IMS. 
	 *			If inIsGM is false IMSLib will use stage endpoint(post 7.0.0.117 version) of IMS.
	 *		   
	 *
	 * @return IMSRef. In case of failure the IMSRef returned is NULL else some non-NULL valid pointer
	 *
	 */
	EXPORT IMSRef IMS_createRef(void* const inClientContext, bool inIsGM);

	/**
    *
    * Description:  This API will return the reference for IMSLib. It will initialize IMSLib, this API will be first call for all the modules that will be using IMSLib.
    *
	*		  This api could be used to hit the client-desired URL for IMS as per the need(e.g. respective end-point urls for Stage, Prod, QA, etc environs).
    *         CNAME provided by IMS team to respective clients can be passed here to target that specific CNAME.                                                                               
    *
    * @param  inClientContext is the context of the client, which will be maintained in IMSRef and later returned in the HTTPCallbackFn (REQUIRED).
    * @param  inIMSEndpoint is the char* argument that will take required IMS URL. Please note that the url needs to be passed in this format ims-na1.adobelogin.com.
    *         If inIMSEndpoint is NULL IMSLib will use production endpoint of IMS. 
    *
    * @return IMSRef. In case of failure the IMSRef returned is NULL else some non-NULL valid pointer
    *
    */

	EXPORT IMSRef IMS_createRefEx( void * const inClientContext, const char *inIMSEndpoint);

	/**
    *
    * Description:  This API will return the reference for IMSLib. It will initialize IMSLib, this API will be first call for all the modules that will be using IMSLib.
	*				This API will pick the IMS endpoint from dispatch table (placed in p7 folder as a part of ASU) on the basis of LEID.
    *
    * @param  inClientContext is the context of the client, which will be maintained in IMSRef and later returned in the HTTPCallbackFn (REQUIRED).
    * @param  inLEID is Licensing ID of the product like V7{}REAP-RS7-Win-PR                                                                    
    *
    * @return IMSRef. In case of failure the IMSRef returned is NULL else some non-NULL valid pointer
    *
    */
	EXPORT IMSRef IMS_createRefForProductVersion( void * const inClientContext, const char* const inLEID);

	
	/**
    *
	* Description:  This API will return the reference for IMSLib. It will initialize IMSLib, this API will be first call for all the modules that will be using IMSLib.
    *
    * @param inClientContext is the context of the client, which will be maintained in IMSRef and later returned in the HTTPCallbackFn (REQUIRED).
	* @param inXMLData is the XML data passed by the caller to IMSLib, content of which are expected as below(REQUIRED).
	* sample inData XML
     <pwaArgs>
        <endPoint>ims-na1.adobelogin.com</endPoint>
        <LEID>Muse-1.0-Win-GM</LEID>
		<userAgent>{"ac":"thor","av":"thor-version-123"}</userAgent>        
		<locale>en_us</locale>
    </pwaArgs>
    * content expectation under tags would be as below
	* <IMSEndPoint> should contain the required IMS endpoint URL intended to be hit. The url needs to be passed in this format ims-na1.adobelogin.com.
    *         This end-point could be used to hit the client-desired URL for IMS as per the need(e.g. respective for Stage, Prod, QA, etc environments).
    *         CNAME provided by IMS team to respective clients can be passed here to target that specific Cname.
    *
	* <LEID> should contain the Licensing ID of the product like V7{}REAP-RS7-Win-GM
    *        Based on the this LEID value, the required IMS URL to hit is picked from the distachTable(placed in P7 folder as a part of ASU)
	*        Preference is given to <IMSEndPoint> as compared to <LEID> 
	*
	* <userAgent> would contain client name and version as desired by IMS as shown above and is passed-as user-agent to IMS APis
    * <Locale> would contain the desired locale in which any IMS jumpLink's content would be displayed in
	* <userAgent> and <Locale> content is opaque for IMSLib and it doesn't try to look into it. 
	*/
    EXPORT IMSRef IMS_createRefExWithXMLParams( void * const inClientContext, const char *inXMLData );


	/**
	 * Description:  This API will un-initialize IMSLib, this API will be the last call for all the modules that will be using IMSLib.
	 * 
	 * @param  inRef is a non NULL pointer to object created by IMS_CreateRef (REQUIRED).
	 *
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_releaseRef( IMSRef * inRef);
	
	/**
	 * Description: This asynchronous API will be called by all the IMSLib clients to fetch access token using the device token present on the machine. 
	 *
	 * @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	 * @param inClientId is the unique identifier of specific client, for which we are asking for access_token (REQUIRED).
	 * @param inClientSecret The secret code of the client (REQUIRED).
	 * @param inUserAccountGuid  is the user account GUID against which access token needs to be generated (REQUIRED).
	 * @param inServiceAccountGuid is the service account GUID of the user account. This will not be used currently (OPTIONAL).
	 * @param inScope is the comma delimited scopes for which the refresh token was requested (REQUIRED).
	 * @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 *
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_fetchAccessToken(IMSRef inRef, const char* const inClientId, const char* const inClientSecret, const char* const inUserAccountGuid, const char* const inServiceAccountGuid, const char* const inScope, HTTPCallbackFn inCallbackFn);

	/**
	 * Description: This asynchronous API will be called by all the IMSLib clients to fetch access token using the device token present on the machine. 
	 * 
	 * @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	 * @param inXMLData is the XML data passed by the caller to IMSLib, content of which are expected as below(REQUIRED).
	 * sample inData XML
	   <pwaArgs>
		  <clientId>AAM_PRE</clientId> 
		  <clientSecret>c6cab08a-bf58-410c-bb45-39e7f8d1338e</clientSecret> 
		  <userGUID></userGUID>
		  <saoGUID></saoGUID>
		  <scope>openid,creative_cloud,csdse,adobeconnect,carousel</scope>
		  <locale>en_us</locale>
	   </pwaArgs>
	 * @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 *
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_fetchAccessToken2(IMSRef inRef, const char* const inXMLData, HTTPCallbackFn inCallbackFn);


	/**
	 * Description:
	 * This api could be used for 2 purposes
     * 1. Storing deviceToken for later accessToken retrieval using device type grant
	 *    This option would be used when no or invalid device token is present on the machine, device token is to be passed under <deviceToken> tag as mentioned below.
     * 
	 * 2. Retrieving one-time usable accessToken against the authorizationKey retrieved from IMS using the Authorization grant type
	 *	   For using this one-time usable key, please pass the authorization key under the tag <key> instead of <deviceToken> as per the API below. Most of the clients can ignore this option
	 *
	 * This API will be invoked by the UI component which has acquired deviceToken/authorizationKey 
	 * after taking user credentials as input. Using the authentication key this API will generate device token which will be hence persisted on user machine.
	 *
	 * @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	 * @param inData is the XML data passed by AAM to IMSLib (REQUIRED).
	 * sample inData XML
	   <pwaArgs>
		  <clientId>AAM_PRE</clientId> 
		  <clientSecret>c6cab08a-bf58-410c-bb45-39e7f8d1338e</clientSecret> 
		  <deviceToken></deviceToken> OR <key></key>
		  <scope>openid,creative_cloud,csdse,adobeconnect,carousel</scope> 
		  <deviceId>DummyDeviceId</deviceId> 
		  <vulcanId>DummyVulcanId</vulcanId> 
		  <callerId>DummyCallerId</callerId> 
	   </pwaArgs>

	 * @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 *
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_fetchIDPResponseForKey(IMSRef inRef, const char* const inData, HTTPCallbackFn inCallbackFn);

	/**
	 * 
	 * Description: This API will revoke the user authorization on this device.
	 * This will wipe out the access_token and it's associated refresh_token, and will perform a logout operation in Renga.
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param inClientId The unique identifier of the client (OPTIONAL).
	 * @param inClientSecret The secret code of the client (OPTIONAL).
	 * @param inCallbackFn The function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 * 
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_revokeDeviceToken(IMSRef inRef, const char* const inUserAccountGuid, const char* const inClientId, const char* const inClientSecret, HTTPCallbackFn inCallbackFn);

	/**
	* Description: This API will fetch the list of all signInUsers on the machine for a particular service code. Caller should invoke IMS_releaseData API after it has copied outAccounts buffer
	 *				to some location. IMS_releaseData should be called with arguments : IMS_releaseData(inRef, IMS_TYPE_ACCOUNTS, (void*)outAccounts);
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inClientId The unique identifier of the client (REQUIRED).
	* @param outAccounts is the non NULL char pointer. To this pointer IMSLib will allocate NULL terminated XML string, which constitutes of user data (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	
    EXPORT IMS_Status IMS_fetchAccounts(IMSRef inRef, const char* const inClientId, char** outAccounts);

	/**
	* Description: This API will free the data allocated to the inIMSLibData pointer. The memory allocated by IMS_fetchAccounts API can be freed by using this API.
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inDataType is the type of inIMSLibData being passed to this API. Since, current usage of this API is for releasing fetchAccounts data this variable should have value IMS_TYPE_ACCOUNTS (REQUIRED).
	* @param inIMSLibData is the non NULL char pointer. This API will flush the user data allocated to the inIMSLibData pointer (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/

	EXPORT IMS_Status IMS_releaseData(IMSRef inRef, IMS_Data_Type inDataType, void* inIMSLibData);	

	/**
	* Description: This API will store device token and profile data of user for a particular client_id.
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inClientId The unique identifier of the client (REQUIRED).
	* @param inDeviceToken is token obtained after user sign in. (REQUIRED).
	* @param inDeviceID is the machine specific device ID generated for fetching device token. (REQUIRED).
	* @param inUserProfileData is JSON user profile returned by server. (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	EXPORT 	IMS_Status IMS_storeDeviceTokenAndUserProfile(IMSRef inRef, const char* const inClientId, const char* const inDeviceToken, const char* const inDeviceID, const char* const inUserProfileData);

	/**
	* Description: This API will fetch profile data of the last logged in user for a particular client_id. Caller should invoke IMS_releaseData API after it has copied outUserData buffer
	 *				to some location. IMS_releaseData should be called with arguments : IMS_releaseData(inRef, IMS_TYPE_DEFAULT_USER_DATA, (void*)outUserData);
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inClientId The unique identifier of the client (REQUIRED).
	* @param outUserData is the non NULL char pointer. To this pointer IMSLib will allocate NULL terminated XML string, which will constitute user data (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	EXPORT IMS_Status IMS_fetchDefaultUserInfoForClientId(IMSRef inRef, const char* const inClientId, char** outUserData);


	/**
	* Description: This API will fetch user profile given a person guid. Caller should invoke IMS_releaseData API after it has copied outUserData buffer
	 *				to some location. IMS_releaseData should be called with arguments : IMS_releaseData(inRef, IMS_TYPE_DEFAULT_USER_DATA, (void*)outUserData);
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inUserGuid is the guid for the user (REQUIRED).
	* @param outUserData is the non NULL char pointer. To this pointer IMSLib will allocate NULL terminated XML string, which will constitute user data (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	EXPORT IMS_Status IMS_fetchUserProfileData(IMSRef inRef, const char* const inUserGuid, char** outUserData);


	/**
	* Description: This API will persist the proxy credentials passed to it in the local store.
	*
	* @param inProxyUsername is a non NULL and non empty string that signifies proxyUserName. inProxyUsername cannot be an empty string. (REQUIRED).
	* @param inProxyPassword is the non NULL string that signifies proxyPassword. inProxyPassword can be an empty string  (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	EXPORT IMS_Status IMS_setProxyCredentials(const char* const inProxyUsername, const char* const inProxyPassword);

	/**
	* Description: This API will persist the proxy credentials passed to it in the IMSLib's session.
	*
	* @param inProxyUsername is a non NULL and non empty string that signifies proxyUserName. inProxyUsername cannot be an empty string. (REQUIRED).
	* @param inProxyPassword is the non NULL string that signifies proxyPassword. inProxyPassword can be an empty string  (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/

	EXPORT IMS_Status IMS_setProxyCredentialsInIMSLibSession(IMSRef inRef, const char* const inProxyUsername, const char* const inProxyPassword, bool inSaveOnDisk);
	
	/**
	* Description: This API will fetch proxy credentials from the IMSLib's session. Caller should invoke IMS_releaseData API after it has copied outProxyData buffer to some location. IMS_releaseData should be called with arguments : IMS_releaseData(inRef, IMS_TYPE_PROXY_DATA, (void*)outProxyData);
	*
	* @param outProxyData is the non NULL char pointer. To this pointer IMSLib will allocate NULL terminated XML string, which constitutes of proxyData data (REQUIRED).
	* outProxyData will return XML in following format 
	*				<ProxyCredentials>
	*					<proxyUsername></proxyUsername>  Base64Encode data
	*					<proxyPassword></proxyPassword>  Base64Encoded data
	*				</ProxyCredentials>
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/
	EXPORT IMS_Status IMS_getProxyCredentialsFromIMSLibSession(IMSRef inRef, char** outProxyData);

/**
	* Description: This API will fetch continue token for a particular client_id with the access token.
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param inBearerToken is the access token issued by IMS to the calling client (REQUIRED).
	* @param inTargetClientId is the client_id needed for the destination (REQUIRED).
	* @param inRedirectUri is the URL to jump to (OPTIONAL).
	* @param inScope is the scope desired (OPTIONAL).
	* @param inResponseType is the IMS integration type the target has (OPTIONAL).
	* @param inLocale is the locale in the format language_country. This will be used to present the login page in case this is required. Default is en_US (OPTIONAL).
	* @param HTTPCallbackFn inCallbackFn is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*/



	EXPORT IMS_Status IMS_fetchContinueToken(IMSRef inRef, 
											const char* const inBearerToken, 
											const char* const inTargetClientId,
											const char* const inRedirectUri,
											const char* const inScope,
											const char* const inResponseType,
											const char* const inLocale,
											HTTPCallbackFn inCallbackFn);


	/**
	 * Description: This API will peform user sign in with the user credentials passed to it. This API will also save user profile data and device token in the local store.
	 *
	 * @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	 * @param inData is the XML data passed by AAM to IMSLib (REQUIRED).
	 * sample inData XML
	   <pwaArgs>
		  <userName/>
		  <userPassword/>
		  <clientId/>
		  <clientSecret/>
		  <deviceId/>
		  <deviceName/>
		  <scope/>
		  <state/>
		  <redirectUri/>
		  <locale/>
	   </pwaArgs>
	 * @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 *
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */

	EXPORT IMS_Status IMS_EEPLogin(IMSRef inRef, const char* const inData, HTTPCallbackFn inCallbackFn);
	
	/**
	 * 
	 * Description: This API will set a particular user as default for a client_id.
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param inUserAccountGuid is the user account GUID, that indicates a user who needs to be set as default user for a clientId  (REQUIRED).
	 * @param inClientId The unique identifier of the client (REQUIRED).
	 *
	**/

	EXPORT IMS_Status IMS_setUserAsDefaultForClientId(IMSRef inRef, const char* const inUserAccountGuid, const char* const inClientId);

	/**
	 * 
	 * Description: This API will fetch the deviceID already present on user's machine. Caller should invoke IMS_releaseData API after it has copied outDeviceId buffer
	 *				to some location. IMS_releaseData should be called with arguments : IMS_releaseData(inRef, IMS_TYPE_DEVICE_ID, (void*)outDeviceId);
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param outDeviceId is is the non NULL char pointer. To this pointer IMSLib will allocate and assign DeviceID  (REQUIRED).
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *
	**/
	EXPORT IMS_Status IMS_getDeviceID(IMSRef inRef, char ** outDeviceID);

	/**
	 * 
	 * Description: This API will remove a particular user as default for a client_id.
	 * In case the user has logged in only for a specific client, this API will revoke user's deviceToken at server end.
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param inClientId The unique identifier of the client (REQUIRED).
	 * @param inCallbackFn The function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	 * 
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_logOutUserForClient(IMSRef inRef, const char* const inUserAccountGuid, const char* const inClientId, HTTPCallbackFn inCallbackFn);
	
	/**
	 * 
	 * Description: This API will register the clientId against the groupId passed as parameter.
	 *				SSO(sign-in/sign-out) of a user for  all the clientIds against this groupId will be done together as 1 atomic group.
	 *				Please consult the IMSLib(dl-imslib-eng) team before planning to use this exported api.	
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param inClientId The unique identifier of the client (REQUIRED).
	 * @param groupId against which the inClientId is to a part of  (REQUIRED).
	 * 
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_registerClientToGroup(IMSRef inRef,  const char* const inClientId, IMS_Buddy_Group groupId);

	/**
	 * 
	 * Description: This API will deregister the clientId against the groupId passed as parameters.
	 *				SSO(sign-in/sign-out) of a user from deregistered client and the Group against which it deregisters will be independent after successful call to this API
	 *				Please consult the IMSLib(dl-imslib-eng) team before planning to use this exported api.	
	 *
	 * @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	 * @param inClientId The unique identifier of the client (REQUIRED).
	 * @param groupId against which the inClientId is a part of  (REQUIRED).
	 * 
	 * @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	 *		
	 */
	EXPORT IMS_Status IMS_deregisterClientFromGroup(IMSRef inRef,  const char* const inClientId, IMS_Buddy_Group groupId);

	/**
	* Description: This asynchronous API will be called by the IMSLib clients to fetch feature flags for a particular user. If release flag is passed during API call, that will be used to fetch the feature flag otherwise API will first fetch the release flag and then will fetch the feature flag.
	*
	* @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	* @param inClientId is the unique identifier of specific client, for which we are asking for access_token (REQUIRED).
	* @param inAccessToken is the access token provided by IMS (REQUIRED).
	* @param inReleaseFlag is the release flag which need to be passed by the caller if available(OPTIONAL)
	* @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*`
	*/
	EXPORT IMS_Status IMS_fetchFeatureFlags(IMSRef inRef, const char* const inClientID, const char* const inAccessToken, const char* const inReleaseFlag, HTTPCallbackFn inCallbackFn);

	/**
	* Description: This asynchronous API will be called by the IMSLib clients to fetch feature flags for a particular user. If release flag is passed during API call, that will be used to fetch the feature flag otherwise API will first fetch the release flag and then will fetch the feature flag.
	*
	* @param inRef is a non NULL object created by IMS_CreateRef (REQUIRED).
	* @param inClientId is the unique identifier of specific client, for which we are asking for access_token (REQUIRED).
	* @param inAccessToken is the access token provided by IMS (REQUIRED).
	* @param inCallbackFn  is the function pointer to the client callback which will eventually receive response from this API (REQUIRED).
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*`
	*/
	EXPORT IMS_Status IMS_fetchReleaseFlag(IMSRef inRef, const char* const inClientID, const char* const inAccessToken, HTTPCallbackFn inCallbackFn);

	/**
	*
	* Description: This API will take duration and maximum retry count from the caller. If IMS server returns RetryAfter header, IMSLib will keep retrying until either maxRetryCount is reaches or maxRetryDuration being specified expires.
	*  If this API is called with positive(non-zero) maxRetryDurationInMillisecs parameter, IMSlib will assume that it need to take care of the error until the specified number of seconds get passed or max retry count is reached.
	*              Please consult the IMSLib(dl-imslib-eng) team before planning to use this exported api.
	*
	* @param inRef IMSLib session object created by IMS_CreateRef (REQUIRED).
	* @param maxRetryDurationInMillisecs duration in millisecs for which IMSLib will keep trying the call on behalf of caller (REQUIRED).
	* @param maxRetryCount specifies maximum number of times IMSLib should retry the call(REQUIRED)
	*
	* @return IMS_Status code. IMS_SUCCESS in case of success else corresponding error code.
	*
	*/
	EXPORT IMS_Status IMS_setRetryAfterResponseHandlingData(IMSRef inRef, int maxRetryDurationInMillisecs, int maxRetryCount);
	
#endif
