#ifndef PI
#define PI 3.14159265359
#endif

#define SAMPLES 8

  // 0    ~= b | color0
  // 0.25 ~= b | color1
  // 0.50 ~= b | color2
  // 0.75 ~= b | color3
  // 1.0  ~= b | color4
#define COLOR_BREAK0 0.0
#define COLOR_BREAK1 0.05
#define COLOR_BREAK2 0.30
#define COLOR_BREAK3 0.55
#define COLOR_BREAK4 0.80

uniform sampler2D texture_param;
uniform vec2      texture_normalizer;
uniform vec4      fill_color0;
uniform vec4      fill_color1;
uniform vec4      fill_color2;
uniform vec4      fill_color3;
uniform vec4      fill_color4;
uniform vec4      outline_color0;
uniform vec4      outline_color1;
uniform vec4      outline_color2;
uniform vec4      outline_color3;
uniform vec4      outline_color4;

uniform float     outline_width_pixels;
in vec2 v_uv;
layout(location = 0) out vec4 fragColor;

float normpdf(in float x, in float sigma) {
  return 0.39894*exp(-0.5*x*x/(sigma*sigma))/sigma;
}

vec3 checker(vec3 color0, vec3 color1, float checker_scale) {
  vec2 checker_uv = floor(checker_scale * gl_FragCoord.xy * texture_normalizer.xy);
  float checker_signal = mod(checker_uv.x + checker_uv.y, 2.0);
  return mix(color0, color1, checker_signal);
}

float inside_coefficient_single_sample(sampler2D texture_param, in vec2 v_uv, in vec2 texture_normalizer) {
  vec4  center = texture( texture_param,  v_uv );
  if (center.a > 0.0) {
    return 1.0; //inside
  } else {
    return 0.0; //inside
  }
}

vec4 shade_selection(in vec2 uv) {
  float outline_alpha = 0.0;
  float convolved_alpha = 0.0;

  float angle = 0.0;
  vec4 sample_bounds = vec4(0.0, 0.0, 1.0, 1.0);

  vec4  center = texture( texture_param,  uv );
  vec3  mesh_mark_color = center.rgb;

  float center_mesh_id = mesh_mark_color.r;
  // outline width is modulated by mesh_mark_color blue channel (typically should be 0.5 (highlight) or 1.0 (selection))
  float outline_scale = 1.0; // 0.5 + 0.5 * step(0.4, mesh_mark_color.b); //; 0.5, 1.0
  float center_alpha = center.a;
  bool inter_mesh_boundary = false;

  for( int i=0; i<SAMPLES; i++ ){
    angle += 1.0/(float(SAMPLES)/2.0) * PI;
    vec2 test_point_offset = 0.5 * vec2( (outline_width_pixels*texture_normalizer.x)*cos(angle), (outline_width_pixels*texture_normalizer.y)*sin(angle) );
    test_point_offset *= outline_scale;
    vec2 test_point = clamp( uv + test_point_offset, sample_bounds.xy, sample_bounds.zw );

    vec4 sample_lookup = texture( texture_param,  test_point );

    float sampled_mesh_id = sample_lookup.r;
    float sampled_alpha = sample_lookup.a;

    if (center_alpha > 0.0 && sampled_mesh_id > center_mesh_id) {
      //If the thing just checks for inequality, the line on the boundary condition would be twice is thick
      //So we use ">" instead of "!=", to just pick one or the other:
      //   slight shift in continuity of the outline is not very noticeable.
      //TODO: Better way to pick one outline over the other would probably be comparing depth values, but we don't have
      //access to depth buffer in this shader yet.

      if (sample_lookup.a > 0.0) {
        //We are looking at the mesh other than the one at the center pixel.
        sampled_alpha = 0.0;
        inter_mesh_boundary = true;
      } else {
        //We are looking at the empty space, nothing to see here.
        sampled_alpha = 0.0;
      }
    } else {
      // mesh mark color - which do we use? center of the sampling does not necessarily have one
      // let's use the max of all the samples, for lack of a better heuristic. we could potentially
      // use the color closest to the camera
      mesh_mark_color = max(mesh_mark_color, sample_lookup.rgb);
    }
    convolved_alpha += sampled_alpha;
  }

  convolved_alpha /= SAMPLES;

  // "selection_map.frag" is used in first pass of 3D rendering.
  // Itmarks the underlying texture to indicate whether the selection-mask rendering
  // is "Full-mesh" selection or "Partial-mesh" selection.
  // Based on that, the final rendering outline varies:

  vec4 fill_color = fill_color0;
  vec4 outline_color = outline_color0;

  if (mesh_mark_color.b > COLOR_BREAK4) {
    fill_color = fill_color4;
  } else if (mesh_mark_color.b > COLOR_BREAK3) {
    fill_color = fill_color3;
  } else if (mesh_mark_color.b > COLOR_BREAK2) {
    fill_color = fill_color2;
  } else if (mesh_mark_color.b > COLOR_BREAK1) {
    fill_color = fill_color1;
  }

  if (mesh_mark_color.g > COLOR_BREAK4) {
    outline_color = outline_color4;
  } else if (mesh_mark_color.g > COLOR_BREAK3) {
    outline_color = outline_color3;
  } else if (mesh_mark_color.g > COLOR_BREAK2) {
    outline_color = outline_color2;
  } else if (mesh_mark_color.g > COLOR_BREAK1) {
    outline_color = outline_color1;
  }

  outline_color.a = outline_color.a * step(0.1, convolved_alpha); // (1 - pow(1.0-convolved_alpha, 5.0));
  // outline_color.rgb = checker(outline_color.rgb, outline_color.rgb * 0.9, 50.0);

  float inside_coefficient = 0.0;
  if (center_alpha > 0.0 && inter_mesh_boundary == false) {
    inside_coefficient = 1.0;
  }

  vec4 final_color = mix(outline_color, fill_color, inside_coefficient);
  final_color = vec4(final_color.rgb*final_color.a, final_color.a);
  // debug:
  // vec4 final_color = vec4(outline_color.rgb, inside_coefficient); // mix(outline_color, fill_color, inside_coefficient);
  return final_color;
}

void main() {
  fragColor = shade_selection(v_uv);
}
