//-------------------------------------------------------------------------------------//
//    Aero Experience Engine                                                           //
//    Adobe Corporation 2018                                                           //
//                  _____  _____________________ ________                              //
//                 /  _  \ \_   _____/\______   \\_____  \                             //
//                /  /_\  \ |    __)_  |       _/ /   |   \                            //
//               /    |    \|        \ |    |   \/    |    \                           //
//               \____|__  /_______  / |____|_  /\_______  /                           //
//                       \/        \/         \/         \/                            //
//-------------------------------------------------------------------------------------//

#ifndef AERO_SHADER_INCLUDES
#define AERO_SHADER_INCLUDES

vec2 latLongUV(vec3 dir)
{
	vec3 normalizedDir = normalize(dir);
  if (1.0-abs(normalizedDir.y) < 1e-7) {
    return vec2(0.0,(sign(normalizedDir.y)+1.0)/2.0);
  }

	float lon = atan2(-normalizedDir.z, normalizedDir.x);
	float lat = acos(normalizedDir.y);
	
	vec2 radians = vec2(1.0 / (3.141592 * 2.0), 1.0 / 3.141592);	
	return vec2(lon, lat) * radians;
}

vec3 pow3(vec3 v, float p)
{
  return pow(v, vec3(p,p,p));
}

vec3 toGamma(vec3 linearTerm) 
{
  return max(1.055 * pow(linearTerm, vec3(0.416666667, 0.416666667, 0.416666667)) - 0.055, 0);
}

vec3 toLinear(vec3 gammaTerm)
{
  return gammaTerm * (gammaTerm * (gammaTerm * 0.305306011 + 0.682171111) + 0.012522878);
}

vec3 RGBMIBLDecode ( vec4 rgbm ) 
{
  vec3 hdr = vec3(5.0f, 5.0f, 5.0f) * rgbm.rgb * rgbm.aaa;
  return pow(hdr.rgb, vec3(2.2f, 2.2f, 2.2f) );
}

vec3 Schlick(float cosTheta, vec3 F0)
{
	return F0 + (1.0 - F0) * pow(1.0 - cosTheta, 5.0);
}

vec3 Schlick(float cosTheta, vec3 F0, float roughness)
{
	float invR = 1.0 - roughness;
	return F0 + (max(vec3(invR, invR, invR), F0) - F0) * pow(1.0 - cosTheta, 5.0);
}

float GVis(float dotNV, float k)
{
    return 1.0f / max((dotNV*(1.0f - k) + k), 0.1f);
}


vec3 importanceSampleDiffuse(vec2 Xi, vec3 N )
{
	float pi = 3.141591f;
	float invpi = 0.318309f;
    float CosTheta = 1.0-Xi.y;
    float SinTheta = sqrt(1.0-CosTheta*CosTheta);
    float Phi = 2*pi*Xi.x;

    vec3 H;
    H.x = SinTheta * cos( Phi );
    H.y = SinTheta * sin( Phi );
    H.z = CosTheta;

    vec3 UpVector = abs(N.z) < 0.999 ? vec3(0,0,1) : vec3(1,0,0);
    vec3 TangentX = normalize( cross( UpVector, N ) );
    vec3 TangentY = cross( N, TangentX );
    return TangentX * H.x + TangentY * H.y + N * H.z;
}

//------------------------------------------------------------------------------------
// Based on GGX example in:                                                            
// http://blog.selfshadow.com/publications/s2013-shading-course/karis/s2013_pbs_epic_notes_v2.pdf
//------------------------------------------------------------------------------------
vec3 importanceSampleGGX(vec2 Xi, float roughness, vec3 N)
{
    float a = roughness * roughness;

    float pi = 3.141591f;
    float invpi = 0.31830988618379067239521257108191f;

    float Phi = 2.0f * pi * Xi.x;
    float CosTheta = sqrt((1 - Xi.y) / (1 + (a*a - 1) * Xi.y));
    float SinTheta = sqrt(1 - CosTheta * CosTheta);

    vec3 H;
    H.x = SinTheta * cos(Phi);
    H.y = SinTheta * sin(Phi);
    H.z = CosTheta;

    vec3 UpVector = abs(N.z) < 0.999 ? vec3(0, 0, 1) : vec3(1, 0, 0);
    vec3 TangentX = normalize(cross(UpVector, N));
    vec3 TangentY = cross(N, TangentX);

    return TangentX * H.x + TangentY * H.y + N * H.z;
}

// D(h) for GGX.
// http://graphicrants.blogspot.com/2013/08/specular-brdf-reference.html
float specularD(float roughness, float NoH)
{
    float NoH2 = NoH * NoH;
    float r2 = roughness * roughness;
    return r2 / pow(NoH2 * (r2 - 1.0f) + 1.0f, 2.0f);
}

// Unoptimized version from: http://www.filmicworlds.com/images/ggx-opt/optimized-ggx.hlsl
vec3 directSpecularGGX(vec3 N, // Normal
                       vec3 V, // View
                       vec3 L, // Light
                       vec3 F, // Fresnel evaluated F0
                       float roughness,
                       vec3 specularLightColor,
                       float NoL, // Normal dot light
                       float NoV) // Normal dot view
{
    if (NoL <= 0.0f)
        return vec3(0.0f,0.0f,0.0f);

    vec3 H = normalize( V + L );
    float NoH = saturate( dot( N, H ) );
    float LoH = saturate( dot( L, H ) );
    float NoH2 = NoH * NoH;
	float r2 = roughness * roughness;

    // D
    float d = r2 / max((3.14159f * pow(NoH * NoH * (r2 - 1.0) + 1.0, 2.0)), 1e-6f);

    // V
    float v1i = GVis( NoL, r2 );
    float v1o = GVis( NoV, r2 );
    float v = v1i * v1o;

    vec3 specularResult = d * v * NoL * F * specularLightColor;

    return specularResult;
}

// Anisotropic BRDF based on Hable’s stuff
vec3 directSpecularAnisotropicGGX(vec3 N, // Normal
                                  vec3 V, // View Direction
                                  vec3 L, // Light Direction
                                  vec3 F, // Fresnel evaluated F0
                                  float anisotropy,
                                  float roughness,
                                  vec3 specularLightColor,
                                  vec3 X, // Anisotropy direction X (substitute tangent if unknown)
                                  vec3 Y, // Anisotropy direction Y (substitute bitangent if unknown)
                                  float NoL, // Normal dot light
                                  float NoV ) // Normal dot view
{
    if (NoL <= 0.0f)
        return vec3(0.0f,0.0f,0.0f);

    NoV = saturate(NoV);
    vec3 H = normalize( V + L );
    float NoH = saturate( dot( N, H ) );
    float LoH = saturate( dot( L, H ) );
    float NoH2 = NoH * NoH;

    float HoX = dot( H, X );
    float HoY = dot( H, Y );
    float HoX2 = HoX * HoX;
    float HoY2 = HoY * HoY;

    float rx = roughness;
    float r2 = rx * rx;

    float ry = mix( 0, rx, max(0.01f, 1.0f - anisotropy) );
    float rx2 = rx * rx;
    float ry2 = ry * ry;

    // D
    float dDenom = HoX2 / rx2 + HoY2 / ry2 + NoH2;
    float d = (1.0f / 3.14159f) * (1.0f / max(rx * ry,1e-6f)) * (1.0f / max(dDenom * dDenom, 1e-6f));

    // V
    float v1i = GVis( NoL, r2 );
    float v1o = GVis( NoV, r2 );
    float v = v1i * v1o;

    vec3 specularResult = d * v * NoL * F * specularLightColor;

    return specularResult;
}

// hash for vectors
// reference: https://developer.download.nvidia.com/assets/gameworks/downloads/regular/GDC17/RealTimeRenderingAdvances_HashedAlphaTesting_GDC2017_FINAL.pdf
float hash2D(vec2 v)
{
    return fract( 1.0e4 * sin(17.0 * v.x + 0.1 * v.y) * (0.1 + abs( sin(13.0 *
     v.y + v.x)) ));
}

float hash3D(vec3 v)
{
    return hash2D(vec2(hash2D(v.xy), v.z));
}

vec4 RGBMEncode( vec3 color ) {
  vec4 rgbm;
  color *= 1.0 / 5.0;
  rgbm.a = saturate( max( max( color.r, color.g ), max( color.b, 1e-6 ) ) );
  rgbm.a = ceil( rgbm.a * 255.0 ) / 255.0;
  rgbm.rgb = color / (max(rgbm.a, 1e-6f));
  return rgbm;
}

float radicalInverse_VdC(uint bits)
{
     bits = (bits << 16u) | (bits >> 16u);
     bits = ((bits & 0x55555555u) << 1u) | ((bits & 0xAAAAAAAAu) >> 1u);
     bits = ((bits & 0x33333333u) << 2u) | ((bits & 0xCCCCCCCCu) >> 2u);
     bits = ((bits & 0x0F0F0F0Fu) << 4u) | ((bits & 0xF0F0F0F0u) >> 4u);
     bits = ((bits & 0x00FF00FFu) << 8u) | ((bits & 0xFF00FF00u) >> 8u);
     return float(bits) * 2.3283064365386963e-10;
}

vec2 hammersley(uint i, uint N)
{
     return vec2(float(i)/float(N), radicalInverse_VdC(i));
}

// https://github.com/glslify/glsl-inverse/blob/master/index.glsl
// MIT License.
mat4 aero_inverse(mat4 m) {

  float
      a00 = m[0][0], a01 = m[0][1], a02 = m[0][2], a03 = m[0][3],
      a10 = m[1][0], a11 = m[1][1], a12 = m[1][2], a13 = m[1][3],
      a20 = m[2][0], a21 = m[2][1], a22 = m[2][2], a23 = m[2][3],
      a30 = m[3][0], a31 = m[3][1], a32 = m[3][2], a33 = m[3][3],

      b00 = a00 * a11 - a01 * a10,
      b01 = a00 * a12 - a02 * a10,
      b02 = a00 * a13 - a03 * a10,
      b03 = a01 * a12 - a02 * a11,
      b04 = a01 * a13 - a03 * a11,
      b05 = a02 * a13 - a03 * a12,
      b06 = a20 * a31 - a21 * a30,
      b07 = a20 * a32 - a22 * a30,
      b08 = a20 * a33 - a23 * a30,
      b09 = a21 * a32 - a22 * a31,
      b10 = a21 * a33 - a23 * a31,
      b11 = a22 * a33 - a23 * a32,

      det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;

  return mat4(
      a11 * b11 - a12 * b10 + a13 * b09,
      a02 * b10 - a01 * b11 - a03 * b09,
      a31 * b05 - a32 * b04 + a33 * b03,
      a22 * b04 - a21 * b05 - a23 * b03,
      a12 * b08 - a10 * b11 - a13 * b07,
      a00 * b11 - a02 * b08 + a03 * b07,
      a32 * b02 - a30 * b05 - a33 * b01,
      a20 * b05 - a22 * b02 + a23 * b01,
      a10 * b10 - a11 * b08 + a13 * b06,
      a01 * b08 - a00 * b10 - a03 * b06,
      a30 * b04 - a31 * b02 + a33 * b00,
      a21 * b02 - a20 * b04 - a23 * b00,
      a11 * b07 - a10 * b09 - a12 * b06,
      a00 * b09 - a01 * b07 + a02 * b06,
      a31 * b01 - a30 * b03 - a32 * b00,
      a20 * b03 - a21 * b01 + a22 * b00) / det;
}

#if BGFX_SHADER_LANGUAGE_HLSL
     mat3 toMat3(mat4 transform) 
     {
        return (float3x3)(transform);
     }
#else
     mat3 toMat3(mat4 transform) 
     {
        return mat3(transform);
     }
#endif

vec4 toVec4(vec4 baseValue)
{
    return baseValue;
}

vec4 toVec4(vec3 baseValue)
{
    return vec4(baseValue.x, baseValue.y, baseValue.z, 1.0f);
}

vec4 toVec4(vec2 baseValue)
{
    return vec4(baseValue.x, baseValue.y, 0.0f, 1.0f);
}

vec4 toVec4(float baseValue)
{
    return vec4(baseValue, 0.0f, 0.0f, 1.0f);
}

// Linear
vec4 linearToMDR(vec4 sourceColor)
{
    sourceColor.rgb = toGamma(sourceColor.rgb);
    sourceColor = RGBMEncode(sourceColor.rgb);
    return sourceColor;
}

vec4 linearToSRGB(vec4 sourceColor)
{
    sourceColor.rgb = toGamma(sourceColor.rgb);
    return sourceColor;
}

vec4 linearToToneMappedSRGB(vec4 sourceColor)
{
// TODO, pass exposure down to here. Needs
// to account for a full tone mapper too.
    return sourceColor;
}

// MDR
vec4 mdrToLinear(vec4 sourceColor)
{
    sourceColor.rgb = vec3(5.0f, 5.0f, 5.0f) * sourceColor.rgb * sourceColor.aaa;
    sourceColor.rgb = pow(sourceColor.rgb, vec3(2.2f, 2.2f, 2.2f) );
    return sourceColor;
}

vec4 mdrToSRGB(vec4 sourceColor)
{
    sourceColor.rgb = vec3(5.0f, 5.0f, 5.0f) * sourceColor.rgb * sourceColor.aaa;
    return sourceColor;
}

vec4 mdrToToneMappedSRGB(vec4 sourceColor)
{
// TODO, pass exposure down to here. Needs
// to account for a full tone mapper too.
    return sourceColor;
}

vec4 srgbToLinear(vec4 sourceColor)
{
    sourceColor.rgb = toLinear(sourceColor.rgb);
    return sourceColor;
}

vec4 srgbToMDR(vec4 sourceColor)
{
    sourceColor = RGBMEncode(sourceColor.rgb);
    return sourceColor;
}

vec4 srgbToToneMappedSRGB(vec4 sourceColor)
{
// TODO: pass exposure down, needs full tone mapper.
    return sourceColor;
}


// source: http://gamedev.stackexchange.com/questions/59797/glsl-shader-change-hue-saturation-brightness
vec3 hsv_to_rgb(vec3 c) {
  vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
  vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
  return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

vec3 rgb_to_hsv(vec3 c){
  vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
  vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
  vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));

  float d = q.x - min(q.w, q.y);
  float e = 1.0e-10;
  return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}


vec3 ACESFilmic(vec3 _rgb)
{
// Reference:
// ACES Filmic Tone Mapping Curve
// https://knarkowicz.wordpress.com/2016/01/06/aces-filmic-tone-mapping-curve/
float aa = 2.51f;
float bb = 0.03f;
float cc = 2.43f;
float dd = 0.59f;
float ee = 0.14f;
return saturate( (_rgb*(aa*_rgb + bb) )/(_rgb*(cc*_rgb + dd) + ee) );
}

float safe_dot(vec3 u, vec3 v) {
  return clamp(dot(u,v), 0.0, 1.0);
}

struct Material {
  float metalicity;
  float roughness;
  vec3 base_color;
  vec3 F0; // specular color if metal, else specular_level [0.02, 0.08]
};

struct Light {
  vec3 dir;
  vec3 color;
  float intensity;
  float radius;
};

// Schlick approximation
vec3 fresnel_term(float VdH, vec3 F0) {
  return F0 + (vec3(1.0, 1.0, 1.0)-F0) * pow(2.0, (-5.55473*VdH-6.98316)*VdH);
}

float geometry_term(float roughness, vec3 V, vec3 N, vec3 H) {
  // remapped roughness, to be used for the geometry term calculations,
  // per Disney [16], Unreal [3]. N.B. don't do this in IBL
  float roughness_remapped = 0.5 + roughness/2.0;

  // ToDo: surfaces with little roughness still catch a lot of light far from the specular center
  float NdV = safe_dot(N, V);

  float k = pow(roughness_remapped + 1.0, 2.0)/8.0;
  return NdV/((NdV)*(1.0-k)+k);
}

float distribution_term(float NdH, float alpha2) {
  float D_denominator = ((NdH*NdH)*(alpha2-1.0)+1.0);
  return alpha2/(3.141591 * pow(D_denominator, 2.0));
}

vec3 shade(Light light, vec3 V, vec3 N, Material material) {
  float alpha = material.roughness * material.roughness + light.radius;
  alpha = min(1.0, alpha);
  float alpha2 = alpha * alpha;

  vec3 L = light.dir;
  vec3 H = normalize(V+L);

  float VdH = safe_dot(V, H);
  float NdH = safe_dot(N, H);
  float NdL = safe_dot(N, L);
  float NdV = safe_dot(N, V);

  float D = distribution_term(NdH, alpha2);
  float Gl = geometry_term(material.roughness, L, N, H);
  float Gv = geometry_term(material.roughness, V, N, H);
  vec3 F = fresnel_term(VdH, material.F0);

  vec3 specular_contribution = F*(Gl*Gv*D/(4.0*NdL*NdV + 0.000001));
  vec3 diffuse_color = material.base_color * (vec3(1,1,1) - material.F0) * (1.0 - material.metalicity) * 2.0 / 3.141592;

  return (specular_contribution + diffuse_color) * NdL;
}

float sun_distribution_term(float NdH, float alpha2, float cloudiness) {
  float D_denominator = ((NdH*NdH)*(alpha2-1.0)+1.0);
  float falloff_rate = mix(3.0, 1.7, cloudiness);
  float brightness = mix(0.1, 1.0, cloudiness);
  return brightness * alpha2/(3.141592 * pow(D_denominator, falloff_rate));
}

vec3 sun_shade(Light light, vec3 V, vec3 N, float horizonEdge, Material material) {
  float cloudiness = light.radius;
  float alpha = max(material.roughness*material.roughness, 0.25*cloudiness*cloudiness);
  alpha = min(1.0, alpha);
  float alpha2 = alpha * alpha;

  vec3 L = light.dir;
  vec3 H = normalize(V+L);

  float VdH = safe_dot(V, H);
  float NdH = safe_dot(N, H);
  float NdL = safe_dot(N, L);
  float NdV = safe_dot(N, V);

  float D = sun_distribution_term(NdH, alpha2, cloudiness);
  float Gl = geometry_term(material.roughness, L, N, H);
  float Gv = geometry_term(material.roughness, V, N, H);
  vec3 F = fresnel_term(VdH, material.F0);

  float horizon_blend = max(0.01, max(material.roughness, (1.0 - material.metalicity)));
  float horizon_mask = smoothstep(-horizon_blend*0.2, horizon_blend*0.2, horizonEdge);
  float horizon_mask_roughness_lerp = mix(horizon_mask, 1.0, horizon_blend);

  float sun_size =  0.001 * mix(0.02, 3.0, material.roughness);
  float sun_diffusion_mult = mix(1, 0.5, cloudiness);
  float sun_intensity = max(light.intensity,pow(light.intensity,16.0)) * 29.0;
  vec3 disc = 1.0 * vec3_splat(smoothstep(1.0 - sun_size, 1.0 - horizon_blend * sun_size, NdH)) * 2.0*max(0.0,min(0.5-cloudiness,0.5-material.roughness));

  vec3 specular_contribution = F*Gl*Gv*D/(4.0*NdL*NdV + 0.000001);
  vec3 diffuse_color = material.base_color * (vec3_splat(1.0) - material.F0) * (1.0 - material.metalicity) * 2.0 / 3.141592;

  vec3 sun_light = (specular_contribution * sun_diffusion_mult + diffuse_color) * NdL;
  sun_light += disc;
  sun_light *= light.color * sun_intensity;
  sun_light *= horizon_mask_roughness_lerp;
  return sun_light;
}

#define SAMPLE_MDL_SCALAR(name, uv) \
vec4 name##uv = name##MapTransform * vec4(uv,1,1); \
float name##Result = (texture2D(name##Map, name##uv.xy).r)

#define USE_MDL_SAMPLER(name) \
(name##Value.x < 0.0)

#define SAMPLE_MDL_NORMAL(name, uv) \
vec4 name##uv = name##MapTransform * vec4(uv,1,1); \
vec3 name##Result = (2.0 * texture2D(name##Map, name##uv.xy).rgb - vec3_splat(1.0))

#define SAMPLE_MDL_COLOR(name, uv) \
vec4 name##uv = name##MapTransform * vec4(uv,1,1); \
vec4 name##Result = (texture2D(name##Map, name##uv.xy))


// refract with total internal reflection calculation
vec3 refract_full(vec3 I, vec3 N, float eta) {
  float c = -dot(N,I);
  float k = 1.0 - eta * eta * (1.0 - c*c);
  if (k < 0.0)
    return normalize(reflect(I,N));
  else
    return eta * I + (eta * c - sqrt(k)) * N;
}

// refract assuming entry and exit of a unit sphere with IOR eta.
// This is an approximation of true refraction (which would require true ray casting)
// I and N should be unit length, normalized vectors
vec3 double_refract(vec3 I, vec3 N, float eta) {
  vec3 Tfront = refract_full(I, N, 1.0/eta);
  vec3 Nback = normalize(reflect(N, Tfront));
  return refract_full(Tfront, -Nback, eta);
}

float hash12(vec2 p) {
  vec3 p3  = fract(vec3(p.xyx) * 0.1031);
  p3 += dot(p3, p3.yzx + 19.19);
  return fract((p3.x + p3.y) * p3.z);
}

highp float rand(vec2 co) {
  return hash12(co);
}

vec2 normal_to_uv(vec3 N) {
  // This only works if everything is normalized.
  vec3 R = normalize(N);

  // The spherical coordinate math breaks down near vertical, so check within
  // an epsilon to prevent flickering in upward-pointing faces....
  if (1.0-abs(R.y) < 1e-7) {
    return vec2(0.0,(sign(R.y)+1.0)/2.0);
  }
  float PI = 3.141592;
  float theta = atan2(-R.x, R.z) + PI / 2.0;
  theta /= 2.0 * PI;

  float phi = acos(-R.y);
  phi /= PI;
  return vec2(theta, phi);
}

float pack_16(float a, float b){
  b = clamp(b ,0,1.0 - 1.0/100.0) ; 
  a = clamp(a,0,1.0); 
  return floor(a*255.0) + b;
}

vec2 unpack_16(float a){
  vec2 uout;
  uout.y = saturate(fract(a) + 1.0/100.0);
  uout.x = floor(a) / 255.0;
  return uout;
}

#endif // AERO_SHADER_INCLUDES
